<?php

/*
 * This file is part of the Sylius package.
 *
 * (c) Sylius Sp. z o.o.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Sylius\Component\Core\OrderProcessing;

use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Order\Model\OrderInterface as BaseOrderInterface;
use Sylius\Component\Order\Processor\OrderProcessorInterface;
use Sylius\Component\Shipping\Calculator\DelegatingCalculatorInterface;
use Sylius\Component\Shipping\Calculator\UndefinedShippingMethodException;
use Sylius\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

final class ShippingChargesProcessor implements OrderProcessorInterface
{
    /** @param FactoryInterface<AdjustmentInterface> $adjustmentFactory */
    public function __construct(
        private FactoryInterface $adjustmentFactory,
        private DelegatingCalculatorInterface $shippingChargesCalculator,
    ) {
    }

    public function process(BaseOrderInterface $order): void
    {
        /** @var OrderInterface $order */
        Assert::isInstanceOf($order, OrderInterface::class);

        if (!$order->canBeProcessed()) {
            return;
        }

        foreach ($order->getShipments() as $shipment) {
            $shipment->removeAdjustments(AdjustmentInterface::SHIPPING_ADJUSTMENT);

            try {
                $shippingCharge = $this->shippingChargesCalculator->calculate($shipment);
                $shippingMethod = $shipment->getMethod();

                /** @var AdjustmentInterface $adjustment */
                $adjustment = $this->adjustmentFactory->createNew();
                $adjustment->setType(AdjustmentInterface::SHIPPING_ADJUSTMENT);
                $adjustment->setAmount($shippingCharge);
                $adjustment->setLabel($shippingMethod->getName());
                $adjustment->setDetails([
                    'shippingMethodCode' => $shippingMethod->getCode(),
                    'shippingMethodName' => $shippingMethod->getName(),
                ]);

                $shipment->addAdjustment($adjustment);
            } catch (UndefinedShippingMethodException) {
            }
        }
    }
}
