<?php

namespace Payum\Bundle\PayumBundle\Tests;

use Payum\Bundle\PayumBundle\Reply\HttpResponse as SymfonyHttpResponse;
use Payum\Bundle\PayumBundle\ReplyToSymfonyResponseConverter;
use Payum\Core\Exception\LogicException;
use Payum\Core\Reply\Base;
use Payum\Core\Reply\HttpPostRedirect;
use Payum\Core\Reply\HttpRedirect;
use Payum\Core\Reply\HttpResponse;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Response;

class ReplyToSymfonyResponseConverterTest extends TestCase
{
    public function testShouldReturnRedirectResponseIfPayumHttpRedirectReply(): void
    {
        $expectedUrl = '/foo/bar';

        $reply = new HttpRedirect($expectedUrl);

        $converter = new ReplyToSymfonyResponseConverter();

        $response = $converter->convert($reply);

        $this->assertInstanceOf(Response::class, $response);
        $this->assertStringContainsString('Redirecting to /foo/bar', $response->getContent());
        $this->assertSame(302, $response->getStatusCode());

        $headers = $response->headers->all();
        $this->assertArrayHasKey('location', $headers);
        $this->assertNotEmpty($headers['location']);
        $this->assertSame($expectedUrl, $headers['location'][0]);
    }

    public function testShouldReturnResponseIfPayumHttpResponseReply(): void
    {
        $reply = new HttpResponse('theContent');

        $converter = new ReplyToSymfonyResponseConverter();

        $response = $converter->convert($reply);

        $this->assertInstanceOf(Response::class, $response);
        $this->assertSame('theContent', $response->getContent());
        $this->assertSame(200, $response->getStatusCode());
    }

    public function testShouldReturnResponseIfPayumHttpResponseReplyWithCustomStatusCodeAndHeaders(): void
    {
        $reply = new HttpResponse('theContent', 418, [
            'foo' => 'fooVal',
            'bar' => 'bar',
        ]);

        $converter = new ReplyToSymfonyResponseConverter();

        $response = $converter->convert($reply);

        $this->assertInstanceOf(Response::class, $response);
        $this->assertSame('theContent', $response->getContent());
        $this->assertSame(418, $response->getStatusCode());
        $this->assertArrayHasKey('foo', $response->headers->all());
        $this->assertArrayHasKey('bar', $response->headers->all());
    }

    public function testShouldReturnResponseIfPayumHttpPostRedirectReply(): void
    {
        $reply = new HttpPostRedirect('anUrl', [
            'foo' => 'foo',
        ]);

        $converter = new ReplyToSymfonyResponseConverter();

        $response = $converter->convert($reply);

        $this->assertInstanceOf(Response::class, $response);
        $this->assertSame(200, $response->getStatusCode());
        $this->assertSame($reply->getContent(), $response->getContent());
    }

    public function testShouldReturnResponseIfSymfonyHttpResponseReply(): void
    {
        $expectedResponse = new Response('foobar');

        $reply = new SymfonyHttpResponse($expectedResponse);

        $converter = new ReplyToSymfonyResponseConverter();

        $actualResponse = $converter->convert($reply);

        $this->assertSame($expectedResponse, $actualResponse);
    }

    public function testShouldChangeReplyToLogicExceptionIfNotSupported(): void
    {
        $this->expectException(LogicException::class);
        $this->expectExceptionMessage('Cannot convert reply Mock_Base_');
        $notSupportedReply = $this->createMock(Base::class);

        $listener = new ReplyToSymfonyResponseConverter();

        $listener->convert($notSupportedReply);
    }
}
